# Plots the optical form of the TFR following Kourkchi+2020, also described in the VCC 1964 letter
# (submitted). This avoids the considerable uncertainly in the stellar mass calculation since it
# just shows the optical magnitude on the y-axis. It also includes a more sophisticated correction
# for turbulence that doesn't introduce the larger scatter seen if the simpler verison suggested
# in the BTFR code is used. This then provides a robust, independent alternative when assessing
# possible outliers.
# The main disadvantage is that absolute magnitude isn't really a physical parameter so it can't
# be directly combined with the gas. Very low mass galaxies have been shown to have higher line
# widths than the optical TFR predicts, however for ordinary galaxies, this is unlikely to be much
# of an issue.


import os
import numpy
import matplotlib
matplotlib.use('agg')
from matplotlib import pyplot
import matplotlib.pyplot as plt
import math

# Import the k-correction function from the external file
from calc_kcor import calc_kcor
# Syntax : calc_kcor(filter_name, redshift, colour_name, colour_value)


# Change to directory where script is located
abspath = os.path.abspath(__file__)
dname = os.path.dirname(abspath)
os.chdir(dname)


# USER-DEFINED PARAMETERS
# INPUT FILES
# VC1 sample of normal galaxies
virgofilename       = dname+'/InputTables/VC1GalaxiesClean.txt'# AGES Virgo galaxies 

# ASTROPHYSICAL VALUES (NOT ACTAULLY NEEDED, HOLDOVERS FROM EARLIER CODE)
# Correction for helium according to McGaugh (everyone else uses 1.36)
xHeMg = 1.33

# Solar absolute magnitudes
gsol = 5.36
isol = 4.58		# For consistency with Taylor et al. 2011. Other sources say 4.48.

# Inclination angle limit - don't do photometry for galaxies less inclined than this, velocity 
# correction errors too large (a few exceptions may be made on an individual basis in the code
# later on)
# Converting i to radians here means we never have to do the conversion again later on.
imin = math.radians(45.0)

# HI deficiency upper limit - don't do photometry for galaxies more deficient than this, line 
# width too uncertain; again exceptions can be made
HIdefmax = 0.6



# MAIN CODE
# 1) PLOTTING SETUP
# LaTeX like fonts in the figure
plt.rc('font', family='serif')

titlesize = 15
numbersize = 12

from matplotlib.ticker import AutoMinorLocator
XminorLocator = AutoMinorLocator()
YminorLocator = AutoMinorLocator()

# Disable interactive plotting
pyplot.ioff()

# Set axes limits
vmin = 1.0		
vmax = 2.75		
bmin = -11.0	
bmax = -22.0	



# Velocity width correction
def Wm50conv(W20):
	Wm50 = (W20 - 17.7) / 0.988
	
	return Wm50
	

# Calculate redshift
def RedShift(velocity):
	z = (velocity*1000.0) / 299792458.0
	
	return z
	
# Correction for redshift and resolution
def Wredres(Wm50, z, vres):
	Wcorr = (Wm50 / (1 + z)) - 2.0*vres*0.25
	
	return Wcorr
	

# Correction for turbulence
def Wtcorr(Wm50):
	k1 = 100.0
	k2 = 9.0
	
	x = Wm50 / k1
	e = math.exp(-(x**2))
	
	Wmx2 = Wm50**2 + k2**2 * (1 - 2*e) - 2 * Wm50* k2 * (1 - e)
		
	return math.sqrt(Wmx2) if Wmx2 > 0 else math.nan



# Calculate the total integrated S/N according to ALFALFA (Saintonge 2007)
# Not actually used, but might be useful as an extra filter on what to plot
def aasn(totflux, w50, vres, rms):
	
	if w50 < 400.0:
		wsmo = w50 / (2.0*vres)
	if w50 >= 400.0:
		wsmo = 400.0 / (2.0*vres)
			
	intsn = (1000.0*totflux / w50) * (math.sqrt(wsmo) / rms)
	
	return intsn


# READ IN THE DATA
VirgoFile = open(virgofilename,'r')
# STORE ARRAYS OF DATA FOR PLOTTING. "Special" array used for highlighting features of interest
# Optical magnitudes
MgMainSample = []
MgSpecial    = []
# Velocity width
LWMainSample = []
LWSpecial    = []


for line in VirgoFile:
	if '#' in line:
		continue
	else:
		ID, RA, Dec, Velocity, W50_AGES, W20_AGES, Distance, MHI_corrected, m_g_AGES, m_i_AGES, HIdef_AGES, Ag_AGES, Ai_AGES, SDSS_bovera, PFlux_AGES, TotFlux_AGES, rms_AGES, VelCMB = line.split()
		# Eliminate :
		# a, b, MType, OptDia, i_AGES, Mstar_orig, OptRA, OptDec

		# If HI deficiency unset, use an absurd value to avoid plotting anything
		if HIdef_AGES == '""':
			HIdef_AGES = 10.0

		# Conver b/a to inclination angle
		AA_i = math.acos(float(SDSS_bovera))
			
		# Only do anything for reliable inclination angles, peak S/N > 10, and low deficiencies. 
		# Special exception : plot everything with "Plot" in its ID.
		if ('Plot' in ID) or (AA_i >= imin and float(HIdef_AGES) < HIdefmax and float(PFlux_AGES)/(5.0*float(rms_AGES)) > 2.0):
			
			# For objects outside Virgo, use the ALFALFA distance formulation. As with BTFR code,
			# the Virgo correction isn't needed for other fields : just use velcmb / 70.0
			if float(Distance) > 32.0:
				AA_Distance = float(VelCMB) / 70.0
			if float(Distance) <= 32.0:
				AA_Distance = float(Distance)
				
			W50_AGES = float(W50_AGES)
			W20_AGES = float(W20_AGES)
			
			
			# 1) OPTICAL CONVERSIONS : SDSS raw apparent to external extinction-corrected 
			# absolute magnitudes
			m_ge = float(m_g_AGES) - float(Ag_AGES)
			m_ie = float(m_i_AGES) - float(Ai_AGES)
			
			# k-correction (probably this makes only a miniscule difference)
			redz = RedShift(float(Velocity))
			k_g = calc_kcor('g', redz, 'g - i', m_ge - m_ie)
			k_i = calc_kcor('i', redz, 'g - i', m_ge - m_ie)
			m_gek = m_ge - k_g
			m_iek = m_ie - k_i
		
			# For comparing photometry, use the AA distance (** this part is a holdover from an earlier comparison
			# between AGES and ALFALFA data, but it should do no harm **)
			#M_ge = -(5.0*math.log10(AA_Distance*1000000.0) -5.0 - m_gek)
			#M_ie = -(5.0*math.log10(AA_Distance*1000000.0) -5.0 - m_iek)
		
			# Kourkchi+2020 method for internal extinction correction when W2 magnitudes not available
			DeltaMg = 0.73*(m_gek - m_iek) - 0.45
			Cg = m_gek - DeltaMg
			
			# Convert this pseduo(apparent)magntidue to a pseudo(absolute)magnitude
			M_geks = -(5.0*math.log10(AA_Distance*1000000.0) -5.0 - Cg)
			
			
			# 2) VELOCITY CORRECTIONS			
			# First correct for W20 to this W50m thing
			W_AGES = Wm50conv(W20_AGES)
			
			# Next correct for redshift and resolution
			redz = RedShift(float(Velocity))
			W_AGES2 = Wredres(W_AGES, redz, 10.6)
			
			# Next turbulence
			W_AGES3 = Wtcorr(W_AGES2)
			
			# Finally inclination
			W_AGES4 = W_AGES3 / math.sin(AA_i)
			if W_AGES4 > 0.0:
				logVcorr = math.log10(W_AGES4)
			else:
				logVcorr = None
			
			# Set the array values we'll use for plotting
			if logVcorr is not None:
				LWMainSample.append(logVcorr)
				MgMainSample.append(M_geks)
						
			
			# SPECIALS : highlight the VLA proposed objects separately
			if 'PlotAGESVC1_275' in ID:
				print('Target :', logVcorr, M_geks)
				LWSpecial.append(logVcorr)
				MgSpecial.append(M_geks) #
						
								
			
VirgoFile.close()


# K20 relation, here showing the main relation and +/- 1, 2 sigma scatter
kstn = []
klow = []
khgh = []
k2sigl = []
k2sigh = []

khgh2 = []
kwdt = []

# K20 pseudomagnitude version
for vc in [10.0,575.0]:
	logvc = math.log10(vc)
	kwdt.append(logvc)
	
	mg = -8.04*(logvc - 2.5) - 20.18
	
	kstn.append(mg)
	klow.append(mg - 0.48)
	khgh.append(mg + 0.48)
	
	k2sigl.append(mg - 2.0*0.48)
	k2sigh.append(mg + 2.0*0.48)
	
	# Special 6 sigma line, not used here	
	khgh2.append(mg - 6.0*0.48)



# MAKE THE PLOT
fig=pyplot.figure()
fig.set_size_inches(800.0/100.0,800.0/100.0)

pyplot.xlim((vmin,vmax))
pyplot.ylim((bmin,bmax))
pyplot.xlabel('log(W$_{corr}$) / km s$^{-1}$', fontsize=titlesize)
pyplot.ylabel('M$_{g}$', fontsize=titlesize)

		
plot = fig.add_subplot(111)
plot.tick_params(axis='both', which='major', labelsize=numbersize)
			
ax = plt.axes()
ax.xaxis.set_minor_locator(XminorLocator)
ax.yaxis.set_minor_locator(YminorLocator)
			
ax.tick_params(bottom=True, left=True, top=True, right=True, which='major', length=10)
ax.tick_params(bottom=True, left=True, top=True, right=True, which='minor', length=5)



# K20 line
pyplot.plot(kwdt, kstn, color='black', linewidth=0.5)

# 1 sigma
pyplot.plot(kwdt, klow, color='black',linestyle='--', dashes=(10, 10), linewidth=1.0)
pyplot.plot(kwdt, khgh, color='black',linestyle='--', dashes=(10, 10), linewidth=1.0)

# 2 sigma
pyplot.plot(kwdt,k2sigh, color='black',linestyle='dotted', dashes=(1, 7), linewidth=1.0)
pyplot.plot(kwdt,k2sigl, color='black', linestyle='dotted', dashes=(1, 7), linewidth=1.0)

# Special multi-sigma line
#pyplot.plot(kwdt, khgh2, color='black',linestyle='--', dashes=(10, 10), linewidth=1.0)


# Plot the AGES sample
plt.scatter(LWMainSample, MgMainSample, facecolors='black', edgecolors='black', s=50)

# Plot any special targets
#plt.scatter(LWSpecial[0], MgSpecial[0], facecolors='orange', edgecolors='none', s=100)


pyplot.savefig('OpticalTFR.png',dpi=100,facecolor='white',bbox_inches='tight')

pyplot.close('all')






